package com.ebm_ws.infra.bricks.util;

import java.lang.reflect.Array;
import java.lang.reflect.GenericArrayType;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

public class ReflectionHelper
{
	/**
	 * Returns the raw Class from the Java 5 Type
	 * @param iType
	 * @return
	 */
	public static Class type2Class(Type iType)
	{
		if(iType instanceof Class)
			return (Class)iType;
		if(iType instanceof ParameterizedType)
			return (Class)((ParameterizedType)iType).getRawType();
		if(iType instanceof GenericArrayType)
		{
			Type componentType = ((GenericArrayType)iType).getGenericComponentType();
			Class componentClass = type2Class(componentType);
			return Array.newInstance(componentClass, 0).getClass();
		}
		return null;
	}
	/**
	 * Returns the equivalent non-primitive class
	 * @param iClass
	 * @return
	 */
	public static Class getNonPrimitiveClass(Class iClass)
	{
		if(!iClass.isPrimitive())
			return iClass;
		if(iClass == Integer.TYPE)
			return Integer.class;
		if(iClass == Boolean.TYPE)
			return Boolean.class;
		if(iClass == Double.TYPE)
			return Double.class;
		if(iClass == Float.TYPE)
			return Float.class;
		if(iClass == Short.TYPE)
			return Short.class;
		if(iClass == Byte.TYPE)
			return Byte.class;
		return null;
	}
	/**
	 * Checks if class1 is assignable as class2.
	 * Manages primitive types.
	 * @param iClass1
	 * @param iClass2
	 * @return
	 */
	public static boolean isAssignable(Class iClass1, Class iClass2)
	{
		// --- change primitive type to non-primitive
		if(iClass1 == null || iClass2 == null)
			return false;
		
		// --- treat integer types
		if(isIntegerLike(iClass1) && isIntegerLike(iClass2))
			return true;
		
		iClass1 = getNonPrimitiveClass(iClass1);
		iClass2 = getNonPrimitiveClass(iClass2);

		return iClass1.isAssignableFrom(iClass2);
	}
	/**
	 * Determines the highest compatible class between all classes.
	 * @param iClasses
	 * @return
	 */
	public static Class getBestCompatibleClass(Class[] iClasses)
	{
		if(iClasses == null)
			return null;
		if(iClasses.length == 0)
			return Object.class;
		Class c = iClasses[0];
		if(c == Object.class)
			return Object.class;
		for(int i=1; i<iClasses.length; i++)
		{
			while(!iClasses[i].isAssignableFrom(c))
			{
				c = c.getSuperclass();
				if(c == Object.class)
					return Object.class;
			}
		}
		return c;
	}
	/**
	 * Returns the component type if the given type is of vector type (List or Array)
	 * @param iClass
	 * @return
	 */
	public static Class getVectorElementClass(Type iType)
	{
		Type t = getVectorElementType(iType);
		if(t == null)
			return null;
		return type2Class(t);
	}
	public static Type getVectorElementType(Type iType)
	{
		if(iType instanceof Class)
		{
			Class c = (Class)iType;
			if(c.isArray())
				return c.getComponentType();
			if(List.class.isAssignableFrom(c))
				// --- no way to know the element type
				return Object.class;
			// --- other types are not supported
			return null;
		}
		else if(iType instanceof ParameterizedType)
		{
			ParameterizedType t = (ParameterizedType)iType;
			Class c = (Class)t.getRawType();
			if(c.isArray())
				return c.getComponentType();
			if(List.class.isAssignableFrom(c))
				return t.getActualTypeArguments()[0];
			// --- other types are not supported
			return null;
		}
		else if(iType instanceof GenericArrayType)
		{
			Type componentType = ((GenericArrayType)iType).getGenericComponentType();
			return componentType;
		}
		
		// Collection? Iterable? Enumeration? ResultSet?
		return null;
	}
	/**
	 * Determines whether the given type is vector (array, list, ...)
	 * @param iType
	 * @return
	 */
	public static boolean isVector(Type iType)
	{
		return getVectorElementClass(iType) != null;
	}
	/**
	 * Returns a List object from any vector type
	 * @param iVector
	 * @param copy
	 * @return
	 */
	public static List obj2List(Object iVector/*, boolean copy*/)
	{
		if(iVector == null)
			return null;
		if(iVector.getClass().isArray())
		{
			int len = Array.getLength(iVector);
			ArrayList ret = new ArrayList(len);
			for(int i=0; i<len; i++)
				ret.add(Array.get(iVector, i));
			return ret;
		}
		if(iVector instanceof List)
		{
			/*
			if(copy)
			{
				ArrayList ret = new ArrayList((List)iVector);
				return ret;
			}
			*/
			return (List)iVector;
		}
		return null;
	}
	/**
	 * Transforms a list into the targeted vector type
	 * @param iList
	 * @param iTarget
	 * @return
	 */
	public static Object vect2obj(List iList, Class iTarget)
	{
		if(iList == null)
			return null;
		if(iTarget.isArray())
		{
			Object ret = Array.newInstance(iTarget.getComponentType(), iList.size());
			for(int i=0; i<iList.size(); i++)
				Array.set(ret, i, iList.get(i));
			return ret;
		}
		if(List.class.isAssignableFrom(iTarget))
		{
			if(iTarget == iList.getClass() || iTarget == List.class)
				// --- use this object
				return iList;
            try
            {
            	List ret = (List)iTarget.newInstance();
    			ret.addAll(iList);
    			return ret;
            }
            catch(InstantiationException e)
            {
	            e.printStackTrace();
            }
            catch(IllegalAccessException e)
            {
	            e.printStackTrace();
            }
		}
        return null;
	}
	public static Object obj2Array(Object iVector/*, boolean copy*/)
	{
		if(iVector == null)
			return null;
		if(iVector.getClass().isArray())
		{
			/*
			if(copy)
			{
				int l = Array.getLength(iVector);
				Object ret = Array.newInstance(iVector.getClass().getComponentType(), l);
				System.arraycopy(iVector, 0, ret, 0, l);
				return ret;
			}
			*/
			return iVector;
		}
		if(iVector instanceof List)
		{
			return ((List)iVector).toArray();
		}
		return null;
	}
	public static Object array2obj(Object iArray, Class iTarget)
	{
		if(iArray == null)
			return null;
		int l = Array.getLength(iArray);
		if(iTarget.isArray())
		{
			if(iArray.getClass() == iTarget)
				return iArray;
			Object ret = Array.newInstance(iTarget.getComponentType(), l);
			System.arraycopy(iArray, 0, ret, 0, l);
			return ret;
		}
		if(List.class.isAssignableFrom(iTarget))
		{
            try
            {
            	List ret = (List)iTarget.newInstance();
            	for(int i=0; i<l; i++)
            		ret.add(Array.get(iArray, i));
    			return ret;
            }
            catch(InstantiationException e)
            {
	            e.printStackTrace();
            }
            catch(IllegalAccessException e)
            {
	            e.printStackTrace();
            }
		}
        return null;
	}
	public static boolean isString(Class iClass)
	{
		return iClass == String.class;
	}
	public static boolean isIntegerLike(Class iClass)
	{
		return iClass == int.class || iClass == Integer.class || iClass == short.class || iClass == Short.class || iClass == long.class || iClass == Long.class;
	}
	public static boolean isBoolean(Class iClass)
	{
		return iClass == boolean.class || iClass == Boolean.class;
	}
	public static boolean isDate(Class iClass)
	{
		return iClass == Date.class || iClass == Calendar.class;
	}
	public static Object parse2Type(String iValue, Class target)
	{
		if(iValue == null)
			return null;
		
		if(target == int.class || target == Integer.class)
		{
			return new Integer(iValue);
		}
		else if(target == short.class || target == Short.class)
		{
			return new Short(iValue);
		}
		else if(target == long.class || target == Long.class)
		{
			return new Long(iValue);
		}
		else if(target == boolean.class || target == Boolean.class)
		{
			return "true".equalsIgnoreCase(iValue);
		}
		else if(target == String.class)
		{
			return iValue;
		}
		
		// ???
		return iValue;
	}
	public static Date obj2Date(Object iValue)
	{
		if(iValue == null)
			return null;
		if(iValue instanceof Date)
			return (Date)iValue;
		if(iValue instanceof Calendar)
			return ((Calendar)iValue).getTime();
		return null;
	}
	public static Object date2Obj(Date iDate, Class iTarget)
	{
		if(iDate == null)
			return null;
		if(iTarget == Date.class)
			return iDate;
		if(iTarget == Calendar.class)
		{
			Calendar cal = Calendar.getInstance();//TODO: ???
			cal.setTime(iDate);
			return cal;
		}
		return null;
	}
	
}
