/*******************************************************************************
 * Copyright (c) 2011 EBM Websourcing.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v2.1
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/old-licenses/gpl-2.0.html
 * 
 * Contributors:
 *     EBM Websourcing - initial API and implementation
 ******************************************************************************/
/**
 * PETALS - PETALS Services Platform. Copyright (c) 2008 EBM Websourcing,
 * http://www.ebmwebsourcing.com/
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version. This library is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 * -------------------------------------------------------------------------
 * $Id$
 * -------------------------------------------------------------------------
 */

package com.ebmwebsourcing.eval.launcher;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import com.ebmwebsourcing.eval.EvalLoggerServer;
import com.ebmwebsourcing.eval.launcher.util.CommandReader;
import com.ebmwebsourcing.eval.launcher.util.Locker;
import com.ebmwebsourcing.eval.launcher.util.SystemExitHook;

/**
 * 
 * Created on 29 janv. 08
 * 
 * @author Christophe HAMERLING, Roland Naudin - eBM WebSourcing
 * @since 1.0
 * 
 */
public class EvalLoggerLauncher implements Launcher {

	protected static final String START_COMMAND = "start";

	protected static final String SHUTDOWN_COMMAND = "shutdown";

	protected static final String VERSION_COMMAND = "version";

	
	protected String banner = " -----------------------------------------------------------\n" +
			"|                                                           |\n" +
			"|                          Eval Logger Server                |\n" +
			"|                                                           |\n" +
			" -----------------------------------------------------------\n";

	/**
	 * The server instance
	 */
	protected EvalLoggerServer server;

	/**
	 * System exit hook used if the command line is used
	 */
	protected SystemExitHook systemExitHook;

	/**
	 * The locker indicating that PEtALS is running
	 */
	protected final Locker locker;

	/**
	 * Default constructor
	 */
	public EvalLoggerLauncher() {
		this.locker = new Locker(new File("."));
	}

	/**
	 * 
	 */
	protected void showBanner() {
		System.out.println();
		System.out
		.println(this.banner);
		System.out.println();
	}
	
	public void setBanner(String banner) {
		this.banner = banner;
	}

	/**
	 * The main program
	 * 
	 * @param args
	 */
	public void launch(String[] args) {
		this.showBanner();
		List<String> command = new ArrayList<String>();
		boolean console = true;

		if (args.length == 0) {
			command.add("start");
		}

		for (String arg : args) {
			command.add(arg);
		}

		try {
			if (command.contains(SHUTDOWN_COMMAND)) {
				System.out.println("Eval Logger is stopping...");
				this.shutdown();

			} else if (command.contains(VERSION_COMMAND)) {
				this.version();

			} else if (command.contains(START_COMMAND)) {
				System.out.println("Eval Logger is starting...");
				this.start();

				// show the commandLine mode if asked
				if (console) {
					this.commandLineMode(this.server);
				}
				if (this.systemExitHook != null) {
					Runtime.getRuntime()
					.removeShutdownHook(this.systemExitHook);
				}
				this.systemExitHook.run();

				System.exit(0);
			} else {
				System.out.println("Command '" + command + "' is unknown");
				this.printUsage();
				System.exit(-1);
			}
		} catch (Throwable e) {
			System.out.println("Command processing error : " + command);
			e.printStackTrace(System.err);
			if (this.systemExitHook != null) {
				Runtime.getRuntime().removeShutdownHook(this.systemExitHook);
			}
			System.exit(-1);
		}

	}

	/*
	 * (non-Javadoc)
	 * 
	 */
	public void start() throws Exception {
		// check if started or not stopped correctly
		if (this.locker.isLocked()) {
			throw new Exception(
					"Can not start the Eval Logger server, remove lock file from Eval Logger root path or stop server");
		}

		this.locker.lock();

		Configuration conf = new ConfigurationImpl();
		
		String host = conf.getHost();
		int port = conf.getPort();
		
		if(host == null || port <= 0) {
			throw new Exception("Wrong host or port parameters!!!");
		}
		
		
		this.server = new EvalLoggerServer(port);
		//this.server.start();
		System.out.println("Eval Logger Server start at: " + "http://" + host + ":" + port);
		this.server.start();
		
		// add a hook if a terminate signal is sent from the command
		// line
		this.systemExitHook = new SystemExitHook(this.locker);
		Runtime.getRuntime().addShutdownHook(this.systemExitHook);

	}

	/*
	 * (non-Javadoc)
	 * 
	 */
	public void shutdown() throws Exception {
		if(this.systemExitHook == null) {
			this.systemExitHook = new SystemExitHook(this.locker);
		} 
		this.systemExitHook.start();
	}

	/*
	 * (non-Javadoc)
	 * 
	 */
	public void version() throws Exception {
		System.out.println("No version");
	}

	/**
	 * Print the launcher usage
	 */
	protected void printUsage() {
		System.out.println("usage:");
		System.out
		.println(" -start                  start the Eval Logger container");
		System.out
		.println(" -shutdowm               shutdown the Eval Logger container");
		System.out
		.println(" -version                get the Eval Logger container version");
	}

	/**
	 * To be overrided for specific message
	 */
	protected void printSpecificStartMessage() {
	}

	/**
	 * Show a command line to interact with Petals. TODO we pass the PetalsAdmin
	 * object to the CommandReader to perform some actions. Find a better way.
	 * 
	 * @param node
	 * 
	 * @throws Exception
	 */
	protected void commandLineMode(EvalLoggerServer node) throws Exception {
		CommandReader console = new CommandReader(node);
		console.read();
	}


}
